# Copyright © The Debusine Developers
# See the AUTHORS file at the top-level directory of this distribution
#
# This file is part of Debusine. It is subject to the license terms
# in the LICENSE file found in the top-level directory of this
# distribution. No part of Debusine, including this file, may be copied,
# modified, propagated, or distributed except according to the terms
# contained in the LICENSE file.

"""Migrate existing Workflow Templates to use runtime_parameters."""

from django.db import migrations
from django.db.backends.base.schema import BaseDatabaseSchemaEditor
from django.db.migrations.state import StateApps

from debusine.client.models import RuntimeParameter


def set_workflowtemplate_runtime_parameters(
    apps: StateApps, schema_editor: BaseDatabaseSchemaEditor
) -> None:
    """Migrate existing Workflow Templates to use runtime_parameters."""
    from debusine.server.workflows.base import Workflow

    WorkflowTemplate = apps.get_model("db", "WorkflowTemplate")

    for workflow_template in WorkflowTemplate.objects.all():
        workflow_cls = Workflow.from_name(workflow_template.task_name)
        for key in workflow_cls.task_data_type.__fields__:
            if key not in workflow_template.static_parameters:
                workflow_template.runtime_parameters[key] = RuntimeParameter.ANY
        workflow_template.save()


class Migration(migrations.Migration):

    dependencies = [
        ("db", "0033_workflowtemplate_add_runtime_parameters"),
    ]

    operations = [
        migrations.RunPython(
            set_workflowtemplate_runtime_parameters, migrations.RunPython.noop
        ),
    ]
